pro bivariate_gaussian_f,x,y,n,rho,sx=sx,sy=sy,plot=plot,pdf=pdf,$
                         student=student

if(n_params() le 0) then begin
    print,'--------------------------------------------------------'
    print,'bivariate_gaussian_f,x,y,N,rho,sx=sx,sy=sy,plot=plot'
    print,'create N pairs x,y following a bivariate Gaussian pdf'
    print,'with correlation coefficient rho, zero mean and unit variances'
    print,'keywords:'
    print,'  sx=sigmax, sy=sigmay  specify standard deviations (defs=1)'
    print,'  /plot -> plot y vs x'
    print,'  plot=[nx,ny,widx,widy] -> plot 2-d histogram of pdf,' 
    print,'  together with theoretical pdf'
    print,'EXAMPLE:'
    print,'bivariate_gaussian_f,x,y,10000,0.5,/plot,sx=2'
    print,'bivariate_gaussian_f,x,y,100000,0.5,plot=[100,100],sx=2'
    print,'student=df  -> use student t with df instead gaussian'
    print,'--------------------------------------------------------'
    return
endif

;------------------------------------------------------------------------
;Bivariate distribution with correlation coefficient rho
;i) Set up the covariance matrix: cov[x,y]=rho * sigma_x * sigma_y
; sigma_x^2      cov[x,y]
; cov[x,y]       sigma_x^2

;ii) Find the eigenvalues and eigenvectors of covariance
;matrix. Combine eigenvectors (column vectors) to a matrix T

;iii) Create independent variable x' and y', using variances
;equal to the two eigenvalues

;iv) Finally, (x,y) pairs are then obtained with
;    transfomation (x,y) =  T (x', y')
;where T is formed from the eigenvectors
;------------------------------------------------------------------------

;defaults:
  sigmax=1.
  sigmay=1.
  if(keyword_set(sx)) then sigmax=sx
  if(keyword_set(sy)) then sigmay=sy

  if(keyword_set(student)) then begin
      df=student   
      sigmax=sqrt(df/(df-2))
      sigmay=sigmax
  endif
  
  sigmax2=sigmax^2
  sigmay2=sigmay^2
  cov=rho*sigmax*sigmay
  
  cc=fltarr(2,2)
  cc(0,0)=sigmax2 & cc(1,0)=cov & cc(0,1)=cov & cc(1,1)=sigmay2

;obtain eigenvalues & eigenvectors  j=0,1
;        lambda_j    ee(0,j) and ee(1,j)
;Using IDL function EIGENQL (for real symmetric matrix)
  lambda=eigenql(cc,eigenvectors=ee)

;calculate two independent variables xp,yp

if(not keyword_set(student)) then begin
  xp=randomn(seed,n)*sqrt(lambda(0))
  yp=randomn(seed,n)*sqrt(lambda(1)) 
endif

if(keyword_set(student)) then begin

;    print,lambda
    sigmax2=lambda(0)
    dfx=2*sigmax2/(sigmax2-1)
    x=randomu(seed,n) & xp=x*0.

for i=0,n_elements(x)-1 do begin & xp(i)=T_cvf(x(i),dfx) & endfor

    sigmax2=lambda(1)
    dfx=2*sigmax2/(sigmax2-1)
    x=randomu(seed,n) & yp=x*0.
for i=0,n_elements(x)-1 do begin & yp(i)=T_cvf(x(i),dfx) & endfor
    
endif

;transform these to x,y pairs
  x=ee(0,0)*xp+ee(0,1)*yp
  y=ee(1,0)*xp+ee(1,1)*yp

;--------------------------------------------------------------------
;
  if(keyword_set(plot)) then begin

;check by calculating Pearson coefficient r
      r=correlate(x,y)
      print,r

      title='BIVARIATE GAUSSIAN, rho='+string(rho,'(f8.4)')
      widx=5
      widy=5
      if(keyword_set(student)) then begin
          title='BIVARIATE STUDENT, df, rho='$
            +string(df,'(f8.4)')+string(rho,'(f8.4)')
          widx=10
          widy=10
      endif
      
      if(n_elements(plot) lt 2) then begin
          nwin
          plot,x,y,xr=[-1.,1.]*widx,xs=1,yr=[-1.,1.]*widy,$
            ys=1,/iso,psym=3,title=title
      endif
      
;-------------------------------------------------------------------
;check the 2D density distribution ?
;-------------------------------------------------------------------

      if(n_elements(plot) ge 2) then begin
          nx=plot(0)
          ny=plot(1)
          if(n_elements(plot) ge 4) then begin
              widx=plot(2)
              widy=plot(3)
          endif
;numerical pdf         
          dx=2.*widx/nx
          dy=2.*widy/ny
          histo=hist_2d(x,y,bin1=dx,bin2=dy,$
                        min1=-widx,max1=widx,min2=-widy,max2=widy)          
          pdf=1.*histo(0:nx-1,0:ny-1)/n_elements(x)/dx/dy
          

;theoretical pdf
          xarg=-widx+(findgen(nx)+0.5)/nx*2*widx
          yarg=-widy+(findgen(ny)+0.5)/ny*2*widy
          xt=pdf*0.
          yt=pdf*0.
          for i=0,n_elements(xarg)-1 do begin
              for j=0,n_elements(yarg)-1 do begin
                  xt(i,j)=xarg(i)
                  yt(i,j)=yarg(j)
              endfor
          endfor
          
          pdf_theory=1./(2.*!pi*sigmax*sigmay*sqrt(1.-rho^2))*$
            exp(-0.5/(1.-rho^2)*(xt^2/sigmax2+yt^2/sigmay2-$
                                 2.*rho*xt*yt/sigmax/sigmay))

;plot using contours (certain fractions of maximum)
         
          pdf_max=1./(2.*!pi*sigmax*sigmay*sqrt(1.-rho^2))
          levels=pdf_max*0.5^[10,9,8,7,6,5,4,3,2,1,0]*.999

          nwin
          contour,pdf,xarg,yarg,/iso,lev=levels,$
            c_col=findgen(10)+1,title=title,$
            xr=[-widx,widx],yr=[-widx,widy],xs=1,ys=1
          contour,pdf_theory,xarg,yarg,/iso,lev=levels,$
            c_col=findgen(10)+1,/over,/noerase,c_line=2
      endif

      xyouts,-0.9*widx,0.9*widy,'N='+string(n,'(i8)')
      xyouts,-0.9*widx,0.8*widy,'sigmax='+string(sigmax,'(f6.2)')
      xyouts,-0.9*widx,0.7*widy,'sigmay='+string(sigmay,'(f6.2)')
      xyouts,-0.9*widx,0.6*widy,'r='+string(r,'(f6.2)')

  endif

end
